// Global Variables
let currentAgendas = [];
let deleteAgendaId = null;

// API Configuration
const API_BASE_URL = '../api';
const DB_CONFIG = {
    host: 'localhost',
    user: 'root',
    password: '',
    database: 'queen_florist'
};

// DOM Content Loaded
document.addEventListener('DOMContentLoaded', function() {
    initializeAdmin();
});

// Initialize Admin Panel
function initializeAdmin() {
    setupEventListeners();
    checkLoginStatus();
}

// Setup Event Listeners
function setupEventListeners() {
    // Login form
    const loginForm = document.getElementById('loginForm');
    if (loginForm) {
        loginForm.addEventListener('submit', handleLogin);
    }

    // Logout button
    const logoutBtn = document.getElementById('logoutBtn');
    if (logoutBtn) {
        logoutBtn.addEventListener('click', handleLogout);
    }

    // Add agenda button
    const addAgendaBtn = document.getElementById('addAgendaBtn');
    if (addAgendaBtn) {
        addAgendaBtn.addEventListener('click', () => openAgendaModal());
    }

    // Search and filter
    const searchInput = document.getElementById('searchInput');
    const monthFilter = document.getElementById('monthFilter');
    
    if (searchInput) {
        searchInput.addEventListener('input', filterAgendas);
    }
    
    if (monthFilter) {
        monthFilter.addEventListener('change', filterAgendas);
    }

    // Close modals when clicking outside
    window.addEventListener('click', function(event) {
        const agendaModal = document.getElementById('agendaModal');
        const deleteModal = document.getElementById('deleteModal');
        
        if (event.target === agendaModal) {
            closeAgendaModal();
        }
        if (event.target === deleteModal) {
            closeDeleteModal();
        }
    });
}

// Authentication Functions
function checkLoginStatus() {
    const token = localStorage.getItem('adminToken');
    if (token) {
        showDashboard();
        loadDashboardData();
    } else {
        showLogin();
    }
}

function handleLogin(event) {
    event.preventDefault();
    
    const username = document.getElementById('username').value;
    const password = document.getElementById('password').value;
    
    // Show loading
    showLoading();
    
    // Simulate login (in real app, this would be API call)
    setTimeout(() => {
        if (username === 'admin' && password === 'admin123') {
            // Store token
            localStorage.setItem('adminToken', 'dummy-token-' + Date.now());
            localStorage.setItem('adminUser', username);
            
            hideLoading();
            showDashboard();
            loadDashboardData();
            showToast('Login berhasil!', 'success');
        } else {
            hideLoading();
            showLoginError();
        }
    }, 1000);
}

function handleLogout() {
    localStorage.removeItem('adminToken');
    localStorage.removeItem('adminUser');
    showLogin();
    showToast('Logout berhasil!', 'success');
}

function showLogin() {
    document.getElementById('loginContainer').style.display = 'flex';
    document.getElementById('adminDashboard').style.display = 'none';
    document.getElementById('loginError').style.display = 'none';
    document.getElementById('loginForm').reset();
}

function showDashboard() {
    document.getElementById('loginContainer').style.display = 'none';
    document.getElementById('adminDashboard').style.display = 'block';
    const adminName = document.querySelector('.admin-name');
    if (adminName) {
        adminName.textContent = localStorage.getItem('adminUser') || 'Admin';
    }
}

function showLoginError() {
    const errorElement = document.getElementById('loginError');
    errorElement.style.display = 'flex';
    setTimeout(() => {
        errorElement.style.display = 'none';
    }, 5000);
}

// Dashboard Functions
async function loadDashboardData() {
    try {
        showLoading();
        
        // Load agendas from database
        const agendas = await fetchAgendas();
        currentAgendas = agendas;
        
        // Update stats
        updateStats(agendas);
        
        // Display agendas
        displayAgendas(agendas);
        
        hideLoading();
    } catch (error) {
        console.error('Error loading dashboard data:', error);
        hideLoading();
        showToast('Gagal memuat data', 'error');
    }
}

function updateStats(agendas) {
    const totalAgendas = document.getElementById('totalAgendas');
    const activeAgendas = document.getElementById('activeAgendas');
    const currentMonthAgendas = document.getElementById('currentMonthAgendas');
    
    const currentMonth = new Date().getMonth() + 1; // JavaScript months are 0-11
    
    totalAgendas.textContent = agendas.length;
    activeAgendas.textContent = agendas.filter(a => a.is_active).length;
    currentMonthAgendas.textContent = agendas.filter(a => a.month === currentMonth).length;
}

// Database Functions (simulated - in real app would connect to PHP backend)
async function fetchAgendas() {
    // Simulate database fetch with sample data
    // In real implementation, this would be an API call to PHP backend
    
    const sampleAgendas = [
        { id: 1, month: 9, day: 24, title: "Hari Dokter Indonesia", description: "Peringatan untuk tenaga medis Indonesia", is_active: 1 },
        { id: 2, month: 9, day: 27, title: "Hari Penerbangan Nasional", description: "Memperingati sejarah penerbangan Indonesia", is_active: 1 },
        { id: 3, month: 9, day: 28, title: "Hari Sumpah Pemuda", description: "Memperingati ikrar pemuda Indonesia 1928", is_active: 1 },
        { id: 4, month: 9, day: 30, title: "Hari Keuangan", description: "Hari kesadaran pentingnya literasi keuangan", is_active: 1 },
        { id: 5, month: 10, day: 1, title: "Hari Kesaktian Pancasila", description: "Memperingati jasa Pahlawan Revolusi", is_active: 1 },
        { id: 6, month: 10, day: 10, title: "Hari Kesehatan Jiwa", description: "Meningkatkan kesadaran kesehatan mental", is_active: 1 },
        { id: 7, month: 10, day: 16, title: "Hari Pangan Sedunia", description: "Meningkatkan kesadaran akan masalah pangan", is_active: 1 },
        { id: 8, month: 10, day: 24, title: "Hari Dokter Indonesia", description: "Menghormati jasa para dokter", is_active: 1 },
        { id: 9, month: 10, day: 28, title: "Hari Sumpah Pemuda", description: "Memperingati ikrar pemuda Indonesia", is_active: 1 },
        { id: 10, month: 10, day: 29, title: "Hari Stroke Nasional", description: "Meningkatkan kesadaran pencegahan stroke", is_active: 1 },
        { id: 11, month: 11, day: 10, title: "Hari Pahlawan", description: "Memperingati jasa pahlawan kemerdekaan", is_active: 1 },
        { id: 12, month: 11, day: 25, title: "Hari Guru Nasional", description: "Menghormati jasa para guru", is_active: 1 }
    ];
    
    return new Promise((resolve) => {
        setTimeout(() => resolve(sampleAgendas), 500);
    });
}

async function saveAgendaToDatabase(agendaData) {
    // Simulate database save
    // In real implementation, this would be an API call to PHP backend
    
    return new Promise((resolve) => {
        setTimeout(() => {
            const newId = Math.max(...currentAgendas.map(a => a.id), 0) + 1;
            const newAgenda = { ...agendaData, id: newId };
            resolve(newAgenda);
        }, 500);
    });
}

async function updateAgendaInDatabase(id, agendaData) {
    // Simulate database update
    return new Promise((resolve) => {
        setTimeout(() => {
            resolve({ ...agendaData, id });
        }, 500);
    });
}

async function deleteAgendaFromDatabase(id) {
    // Simulate database delete
    return new Promise((resolve) => {
        setTimeout(() => {
            resolve(true);
        }, 500);
    });
}

// Display Functions
function displayAgendas(agendas) {
    const tableBody = document.getElementById('agendaTableBody');
    const noDataMessage = document.getElementById('noDataMessage');
    
    if (agendas.length === 0) {
        tableBody.innerHTML = '';
        noDataMessage.style.display = 'block';
        return;
    }
    
    noDataMessage.style.display = 'none';
    
    const monthNames = ['Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni', 
                       'Juli', 'Agustus', 'September', 'Oktober', 'November', 'Desember'];
    
    tableBody.innerHTML = agendas.map((agenda, index) => `
        <tr>
            <td>${index + 1}</td>
            <td>${agenda.day}</td>
            <td>${monthNames[agenda.month - 1]}</td>
            <td>${agenda.title}</td>
            <td>${agenda.description || '-'}</td>
            <td>
                <span class="status-badge ${agenda.is_active ? 'status-active' : 'status-inactive'}">
                    ${agenda.is_active ? 'Aktif' : 'Non-aktif'}
                </span>
            </td>
            <td>
                <div class="action-buttons">
                    <button class="btn-action btn-edit" onclick="editAgenda(${agenda.id})" title="Edit">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button class="btn-action btn-delete" onclick="deleteAgenda(${agenda.id})" title="Hapus">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </td>
        </tr>
    `).join('');
}

// Filter Functions
function filterAgendas() {
    const searchTerm = document.getElementById('searchInput').value.toLowerCase();
    const monthFilter = document.getElementById('monthFilter').value;
    
    let filteredAgendas = currentAgendas;
    
    // Filter by search term
    if (searchTerm) {
        filteredAgendas = filteredAgendas.filter(agenda => 
            agenda.title.toLowerCase().includes(searchTerm) ||
            (agenda.description && agenda.description.toLowerCase().includes(searchTerm))
        );
    }
    
    // Filter by month
    if (monthFilter) {
        filteredAgendas = filteredAgendas.filter(agenda => 
            agenda.month == monthFilter
        );
    }
    
    displayAgendas(filteredAgendas);
}

// Modal Functions
function openAgendaModal(agendaId = null) {
    const modal = document.getElementById('agendaModal');
    const modalTitle = document.getElementById('modalTitle');
    const form = document.getElementById('agendaForm');
    
    // Reset form
    form.reset();
    document.getElementById('agendaId').value = '';
    
    if (agendaId) {
        // Edit mode
        const agenda = currentAgendas.find(a => a.id === agendaId);
        if (agenda) {
            modalTitle.textContent = 'Edit Agenda';
            document.getElementById('agendaId').value = agenda.id;
            document.getElementById('agendaDay').value = agenda.day;
            document.getElementById('agendaMonth').value = agenda.month;
            document.getElementById('agendaTitle').value = agenda.title;
            document.getElementById('agendaDescription').value = agenda.description || '';
            document.getElementById('agendaActive').checked = agenda.is_active;
        }
    } else {
        // Add mode
        modalTitle.textContent = 'Tambah Agenda';
    }
    
    modal.style.display = 'block';
    document.body.style.overflow = 'hidden';
}

function closeAgendaModal() {
    const modal = document.getElementById('agendaModal');
    modal.style.display = 'none';
    document.body.style.overflow = 'auto';
}

// CRUD Functions
async function saveAgenda() {
    const form = document.getElementById('agendaForm');
    const formData = new FormData(form);
    
    const agendaData = {
        day: parseInt(document.getElementById('agendaDay').value),
        month: parseInt(document.getElementById('agendaMonth').value),
        title: document.getElementById('agendaTitle').value.trim(),
        description: document.getElementById('agendaDescription').value.trim(),
        is_active: document.getElementById('agendaActive').checked ? 1 : 0
    };
    
    // Validation
    if (!agendaData.day || !agendaData.month || !agendaData.title) {
        showToast('Mohon lengkapi semua field yang wajib diisi', 'error');
        return;
    }
    
    if (agendaData.day < 1 || agendaData.day > 31) {
        showToast('Tanggal harus antara 1-31', 'error');
        return;
    }
    
    showLoading();
    
    try {
        const agendaId = document.getElementById('agendaId').value;
        
        if (agendaId) {
            // Update existing agenda
            await updateAgendaInDatabase(parseInt(agendaId), agendaData);
            
            // Update local data
            const index = currentAgendas.findIndex(a => a.id === parseInt(agendaId));
            if (index !== -1) {
                currentAgendas[index] = { ...agendaData, id: parseInt(agendaId) };
            }
            
            showToast('Agenda berhasil diperbarui!', 'success');
        } else {
            // Add new agenda
            const newAgenda = await saveAgendaToDatabase(agendaData);
            currentAgendas.push(newAgenda);
            
            showToast('Agenda berhasil ditambahkan!', 'success');
        }
        
        // Refresh display
        updateStats(currentAgendas);
        displayAgendas(currentAgendas);
        closeAgendaModal();
        
    } catch (error) {
        console.error('Error saving agenda:', error);
        showToast('Gagal menyimpan agenda', 'error');
    } finally {
        hideLoading();
    }
}

function editAgenda(agendaId) {
    openAgendaModal(agendaId);
}

function deleteAgenda(agendaId) {
    const agenda = currentAgendas.find(a => a.id === agendaId);
    if (agenda) {
        deleteAgendaId = agendaId;
        document.getElementById('deleteAgendaTitle').textContent = agenda.title;
        document.getElementById('deleteModal').style.display = 'block';
        document.body.style.overflow = 'hidden';
    }
}

async function confirmDelete() {
    if (!deleteAgendaId) return;
    
    showLoading();
    
    try {
        await deleteAgendaFromDatabase(deleteAgendaId);
        
        // Remove from local data
        currentAgendas = currentAgendas.filter(a => a.id !== deleteAgendaId);
        
        // Refresh display
        updateStats(currentAgendas);
        displayAgendas(currentAgendas);
        closeDeleteModal();
        
        showToast('Agenda berhasil dihapus!', 'success');
        
    } catch (error) {
        console.error('Error deleting agenda:', error);
        showToast('Gagal menghapus agenda', 'error');
    } finally {
        hideLoading();
        deleteAgendaId = null;
    }
}

function closeDeleteModal() {
    document.getElementById('deleteModal').style.display = 'none';
    document.body.style.overflow = 'auto';
    deleteAgendaId = null;
}

// Utility Functions
function showLoading() {
    document.getElementById('loadingOverlay').style.display = 'flex';
}

function hideLoading() {
    document.getElementById('loadingOverlay').style.display = 'none';
}

function showToast(message, type = 'success') {
    const toast = document.getElementById('toast');
    const toastMessage = document.getElementById('toastMessage');
    
    toast.className = 'toast ' + type;
    toastMessage.textContent = message;
    toast.style.display = 'block';
    
    setTimeout(() => {
        toast.style.display = 'none';
    }, 3000);
}

// API Functions for PHP Backend (to be implemented)
async function callAPI(endpoint, method = 'GET', data = null) {
    const options = {
        method: method,
        headers: {
            'Content-Type': 'application/json',
            'Authorization': 'Bearer ' + localStorage.getItem('adminToken')
        }
    };
    
    if (data && (method === 'POST' || method === 'PUT')) {
        options.body = JSON.stringify(data);
    }
    
    try {
        const response = await fetch(`${API_BASE_URL}/${endpoint}`, options);
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        return await response.json();
    } catch (error) {
        console.error('API call failed:', error);
        throw error;
    }
}

// Export/Import Functions
function exportAgendas() {
    const dataStr = JSON.stringify(currentAgendas, null, 2);
    const dataUri = 'data:application/json;charset=utf-8,'+ encodeURIComponent(dataStr);
    
    const exportFileDefaultName = `queen-florist-agenda-${new Date().toISOString().split('T')[0]}.json`;
    
    const linkElement = document.createElement('a');
    linkElement.setAttribute('href', dataUri);
    linkElement.setAttribute('download', exportFileDefaultName);
    linkElement.click();
}

function importAgendas(event) {
    const file = event.target.files[0];
    if (!file) return;
    
    const reader = new FileReader();
    reader.onload = function(e) {
        try {
            const importedAgendas = JSON.parse(e.target.result);
            
            if (Array.isArray(importedAgendas)) {
                // Validate imported data
                const isValid = importedAgendas.every(agenda => 
                    agenda.day && agenda.month && agenda.title
                );
                
                if (isValid) {
                    currentAgendas = importedAgendas;
                    updateStats(currentAgendas);
                    displayAgendas(currentAgendas);
                    showToast('Data berhasil diimpor!', 'success');
                } else {
                    showToast('Format data tidak valid!', 'error');
                }
            } else {
                showToast('File harus berupa array agenda!', 'error');
            }
        } catch (error) {
            showToast('Gagal membaca file!', 'error');
        }
    };
    
    reader.readAsText(file);
    event.target.value = ''; // Reset file input
}

// Keyboard shortcuts
document.addEventListener('keydown', function(event) {
    // Ctrl/Cmd + N: New agenda
    if ((event.ctrlKey || event.metaKey) && event.key === 'n') {
        event.preventDefault();
        if (document.getElementById('adminDashboard').style.display !== 'none') {
            openAgendaModal();
        }
    }
    
    // Escape: Close modals
    if (event.key === 'Escape') {
        closeAgendaModal();
        closeDeleteModal();
    }
});

// Auto-save draft
let autoSaveTimer;
function setupAutoSave() {
    const form = document.getElementById('agendaForm');
    if (!form) return;
    
    form.addEventListener('input', function() {
        clearTimeout(autoSaveTimer);
        autoSaveTimer = setTimeout(() => {
            // Save form data to localStorage as draft
            const formData = {
                day: document.getElementById('agendaDay').value,
                month: document.getElementById('agendaMonth').value,
                title: document.getElementById('agendaTitle').value,
                description: document.getElementById('agendaDescription').value,
                is_active: document.getElementById('agendaActive').checked
            };
            localStorage.setItem('agendaDraft', JSON.stringify(formData));
        }, 2000);
    });
}

// Initialize auto-save
setupAutoSave();